// pages/api/test/createOnChain.ts
import type { NextApiRequest, NextApiResponse } from "next";
import {
  walletClient,
  encodeCreateTicketArgs,
  canonicalJsonHash,
  CONTRACT_ADDRESS,
} from "@/lib/chain";

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  if (req.method !== "POST") {
    return res.status(405).json({ ok: false, error: "Method not allowed" });
  }

  try {
    // Example dummy ticket JSON (normally comes from your DB in Step 6)
    const ticketJson = {
      title: "Leaking sink in ER room",
      description: "Water under sink cabinet; slow drip",
      location: "ER-12",
      severity: 2,
      department: "Plumbing",
      // Any extra keys are fine; order doesn’t matter due to canonicalization
      attachments: [],
      createdAt: new Date().toISOString(),
    };

    // 1) Hash canonical JSON server-side
    const jsonHash = canonicalJsonHash(ticketJson);

    // 2) Build tx data for createTicket(hash, severity, department)
    const calldata = encodeCreateTicketArgs(
      jsonHash,
      ticketJson.severity,
      ticketJson.department
    );

    // 3) Send transaction from the server wallet (relayer)
    const txHash = await walletClient.sendTransaction({
      to: CONTRACT_ADDRESS,
      data: calldata,
    });

    return res.status(200).json({
      ok: true,
      txHash,
      jsonHash,
      contract: CONTRACT_ADDRESS,
      example: ticketJson,
    });
  } catch (err: any) {
    console.error(err);
    return res
      .status(500)
      .json({ ok: false, error: err?.message ?? "unknown error" });
  }
}
